// Please see credits and usage for usiTwiSlave and TinyWireS in the .h files of 
// those libraries.

#include <avr/sleep.h>
#include <avr/wdt.h>
#include <EEPROM.h>
#include "TinyWireS.h"                  

#define I2C_DEFAULT_ADDR  0x04            // i2c default address (4, 0x04)
#define NUMBER_OF_DIGITS  0x06            // number of 7-segment digits

#define DISPLAY_NUMBER    0x00
#define FLASH_PATTERN     0x10
#define FLASH_PERIOD      0x11

#define SET_I2C_ADDR      0x50

uint8_t i2c_address = I2C_DEFAULT_ADDR;
uint8_t flash_pattern = 0x00;             //A bit sequence where each bit indicates if the digit should be flashed 
uint8_t flash_period = 50;                //The period to turn the digit on/off
bool flash_state = true;                  //False: Indicates the digit is OFF, True: Indicates the digit is ON

uint32_t time = 0;                 //Holds the current time
uint8_t display_byte[6] = {0,0,0,0,0,0};

                    // Pin 1 - Reserved for Reset
uint8_t SER = 3;    // Pin 2 - SER to 74HC595 
uint8_t SRCLK = 4;  // Pin 3 - SRCLK to 74HC595
                    // Pin 4 - Gnd
                    // Pin 5 - SDA
uint8_t RCLK = 1;   // Pin 6 - RCLK to 74HC595
                    // Pin 7 - SCL
                    // Pin 8 - Vcc


void setup()
{
  pinMode(SER,OUTPUT);                                  //Configure PB3 as output
  pinMode(RCLK,OUTPUT);                                 //Configure PB1 as output
  pinMode(SRCLK,OUTPUT);                                //Configure PB4 as output
  
  i2c_address = EEPROM.read(0);
  if ((i2c_address== 0x00)||(i2c_address== 0xff)){
    i2c_address = I2C_DEFAULT_ADDR;
  }

  update_display(true);
  time = millis();

  TinyWireS.begin(i2c_address);      // Initiazlize the I2C Slave mode
  TinyWireS.onReceive(receiveEvent);    // Register the onReceive() callback function

}

void loop()
{
  if(flash_pattern == 0x00){
    return;
  }

  //Check to see if its time to toggle the state of the display
  if (millis() > (time+flash_period)){
    update_display(flash_state);
    flash_state = !flash_state;
    time = millis();
  }
}

void update_display(bool state){
    
  for (uint8_t i = 0; i < NUMBER_OF_DIGITS; i++){                      // Process each byte of data from the master
    if ((bitRead(flash_pattern, i)==1)&&(state == false)) {
      //If the digit is set to flash and it should be off 
      for (uint8_t x = 0; x < 8; x++){                    //Loop for each bit within data byte
        digitalWrite(SER, 0);                             //Set the SER pin based on the bit of data
        digitalWrite(SRCLK, 1);                           //Set SRCLK High
        digitalWrite(SRCLK, 0);                           //Set SRCLK Low
      }
    }else{
      //For all other cases, just display the number 
      for (uint8_t x = 0; x < 8; x++){                    //Loop for each bit within data byte
        digitalWrite(SER, bitRead(display_byte[i], x)); //Set the SER pin based on the bit of data
        digitalWrite(SRCLK, 1);                           //Set SRCLK High
        digitalWrite(SRCLK, 0);                           //Set SRCLK Low
      }
    }
  }     
  digitalWrite(RCLK, 1);                                // Pulse the RLCK to clear any junk data on the output pins.
  digitalWrite(RCLK, 0);   
}

// Gets called when the ATtiny receives an i2c write slave request
// This routine runs from the usiTwiSlave interrupt service routine (ISR)
// so interrupts are disabled while it runs.
void receiveEvent(uint8_t num_bytes)
{
  uint8_t master_bytes = num_bytes - 2;

  uint8_t command = TinyWireS.receive();
  uint8_t value = TinyWireS.receive();

  switch (command){
    case DISPLAY_NUMBER:
      for (uint8_t i = 0; i < master_bytes; i++){           // Process each byte of data from the master
        display_byte[i] = TinyWireS.receive();
      }     
      update_display(true);
    break;

    case FLASH_PATTERN:                                     //Sets the bit sequence of which digits to flash
      flash_pattern = value;
      
      if (flash_pattern == 0x00){                           //Make sure that the display is ON, if we are stopping the flash mode 
        flash_state = true;
        update_display(flash_state);
      }
      break;

    case FLASH_PERIOD:                                      //Sets the flash period
      flash_period = value;
      break;

    case SET_I2C_ADDR:                                      //Set the I2C address of the module
      if ((value > 0) && (value < 128)){
        EEPROM.write(0,value);
      }
      break;
  }

}

