//----------------------------------------------------
// Developed By: MakingSense_otw
//----------------------------------------------------
// Description:
// Rolls one or more dice and displays the animation
// using the Adafruit_SSD1306 library.
//
// Date: Feb 18th 2024
// Version: 1.0
//----------------------------------------------------

#include "dice.h"
#include <Adafruit_SSD1306.h>

#ifdef ALLOW_DICE_TEXT
#include <Fonts/FreeSansBold9pt7b.h>
#include <Fonts/FreeSansBold12pt7b.h>
#include <Fonts/FreeSansBold18pt7b.h>
#endif

Dice::Dice(){
}

bool Dice::begin(Adafruit_SSD1306 *display, bool solid){
    // Store the references & values for the parameters
  _display = display;
  _solid = solid;
  
  clearGraphics();
  _display->setTextSize(1);
  _display->display();

  return true;
}

void Dice::setNumber(int numberOfDice){
  _numberOfDice = numberOfDice;
}

void Dice::setDiceType(int diceType){
#ifdef ALLOW_DICE_TEXT
  if (diceType > 0 && diceType <= 99){
    _minDiceValue = 1;
    _maxDiceValue = diceType;
  }else if (diceType == 100){
    _minDiceValue = 0;
    _maxDiceValue = 99;
  }
  else
  {
    _minDiceValue = 1;
    _maxDiceValue = 6;
  }
#else
  if (diceType > 0 && diceType <= 6){
    _minDiceValue = 1;
    _maxDiceValue = diceType;
  }
#endif
}

int Dice::getValue(){
  return random(_minDiceValue,_maxDiceValue+1);
}

void Dice::roll(){
  int offSet = 0;
  int remainingSpace = 0;

  if (randomized==false){
    randomSeed(millis());
    randomized = true;
  }
  int rollDelay = 220;

  for (int index = 0; index < _numberOfRollsBeforeStopping; index++)
  {
    
    clearGraphics();
    switch(_numberOfDice){
      case 1:
        drawDiceImage(40, _yPositionDice, X_LARGE_DICE, getValue(), _solid);
        break;

      case 2:
        drawDiceImage(14, _yPositionDice, X_LARGE_DICE, getValue(), _solid);
        drawDiceImage(74, _yPositionDice, X_LARGE_DICE, getValue(), _solid);
        break;

      case 3: 
        remainingSpace = 128 - (LARGE_DICE*3);
        offSet = remainingSpace/2;

        drawDiceImage(0, _yPositionDice, LARGE_DICE, getValue(), _solid);
        drawDiceImage(LARGE_DICE+offSet, _yPositionDice, LARGE_DICE, getValue(), _solid);
        drawDiceImage((LARGE_DICE+offSet)*2, _yPositionDice, LARGE_DICE, getValue(), _solid);
        break;

      case 4: 
        remainingSpace = 128 - (MEDIUM_DICE*4);
        offSet = remainingSpace/3;
        
        drawDiceImage(0, _yPositionDice, MEDIUM_DICE, getValue(), _solid);
        drawDiceImage(MEDIUM_DICE+offSet, _yPositionDice+15, MEDIUM_DICE, getValue(), _solid);
        drawDiceImage((MEDIUM_DICE+offSet)*2, _yPositionDice, MEDIUM_DICE, getValue(), _solid);
        drawDiceImage((MEDIUM_DICE+offSet)*3, _yPositionDice+15, MEDIUM_DICE, getValue(), _solid);
        break;

      case 5: 
        remainingSpace = 128 - (SMALL_DICE*5);
        offSet = remainingSpace/4;

        drawDiceImage(0, _yPositionDice, SMALL_DICE, getValue(), _solid);
        drawDiceImage(SMALL_DICE+offSet, _yPositionDice+20, SMALL_DICE, getValue(), _solid);
        drawDiceImage((SMALL_DICE+offSet)*2, _yPositionDice, SMALL_DICE, getValue(), _solid);
        drawDiceImage((SMALL_DICE+offSet)*3, _yPositionDice+20, SMALL_DICE, getValue(), _solid);
        drawDiceImage((SMALL_DICE+offSet)*4, _yPositionDice, SMALL_DICE, getValue(), _solid);
        break;
    }
    _display->display();
    
    delay(rollDelay);
    rollDelay = rollDelay - 20;
  }

}

void Dice::clearGraphics (void){
  _display->fillRect(0,16,127,48, SSD1306_BLACK);
}

// Draws the image of the dice
void Dice::drawDiceImage(int x, int y, int size, int value, bool solid){
  //There is a margin around the edge
  //Then the middle is split in to three points
   
  int margin = size/4;
  int diceRadius = size /6;
  int dotRadius = size/10;

  int diceSegment = (size - (margin * 2))/2;
    
  // Set spacing of the dots on the dice
  int offset0 = margin;
  int offset1 = margin + diceSegment;
  int offset2 = margin + (diceSegment *2);

  // Set position of each on the dots on the dice
  int leftTopX = x+offset0;
  int leftTopY = y+offset0;

  int leftMiddleX = x+offset0;
  int leftMiddleY = y+offset1;

  int leftBottomX = x+offset0;
  int leftBottomY = y+offset2;

  int middleMiddleX = x+offset1;
  int middleMiddleY = y+offset1;

  int rightTopX = x+offset2;
  int rightTopY = y+offset0;

  int rightMiddleX = x+offset2;
  int rightMiddleY = y+offset1;

  int rightBottomX = x+offset2;;
  int rightBottomY = y+offset2;;

  drawDiceShape(x, y, size, diceRadius, solid);

  if (_maxDiceValue <= 6){
    if(value == 2 ||value == 3 ||value == 4 ||value == 5 ||value == 6){
      drawDiceDots(leftTopX, leftTopY, dotRadius, solid); // Left Top
      drawDiceDots(rightBottomX, rightBottomY, dotRadius, solid); // Right Bottom
    }

    if(value == 6){
      drawDiceDots(leftMiddleX, leftMiddleY, dotRadius, solid); // Left Middle
      drawDiceDots(rightMiddleX, rightMiddleY, dotRadius, solid); // Right Middle
    }

    if(value == 4 ||value == 5 ||value == 6){
      drawDiceDots(leftBottomX, leftBottomY, dotRadius, solid); // Left Bottom
      drawDiceDots(rightTopX, rightTopY, dotRadius, solid); // Right Top
    }

    if(value == 1 ||value == 3 ||value == 5){
      drawDiceDots(middleMiddleX, middleMiddleY, dotRadius, solid); // Middle
    }
  } 
  #ifdef ALLOW_DICE_TEXT
  else 
  {
    drawTextNumber(x, y, size, value, solid);
  }
  #endif
}

#ifdef ALLOW_DICE_TEXT
void Dice::drawTextNumber (int x, int y, int size, int value, bool solid){


  char charArray[6];
  itoa(value,charArray,10);
  
  if (solid){
    _display->setTextColor(SSD1306_BLACK);
  }else{
    _display->setTextColor(SSD1306_WHITE);
  }

  switch (size){
    case SMALL_DICE:
      _display->setFont(&FreeSansBold9pt7b);
      if (value < 10){
        _display->setCursor(x+7,y+18);
        _display->write(charArray);
      }else if (value>=10 && value <=19){
        _display->setCursor(x+1,y+18);
        _display->write(charArray[0]);
        _display->setCursor(x+11,y+18);
        _display->write(charArray[1]);
      }else{
        _display->setCursor(x+2,y+18);
        _display->write(charArray[0]);
        _display->setCursor(x+12,y+18);
        _display->write(charArray[1]);
      }
      
      break;

    case MEDIUM_DICE:
      _display->setFont(&FreeSansBold12pt7b);
      if (value < 10){
        _display->setCursor(x+8,y+21);
        _display->write(charArray);
      }else if (value >=10 && value <=19){
        _display->setCursor(x+1,y+21);
        _display->write(charArray[0]);
        _display->setCursor(x+12,y+21);
        _display->write(charArray[1]);
      }
      else{
        _display->setCursor(x+1,y+21);
        _display->write(charArray);
      }
      
      break;

    case LARGE_DICE:
      _display->setFont(&FreeSansBold18pt7b);
      if (value < 10){
        _display->setCursor(x+11,y+31);
        _display->write(charArray);
      }else if (value>= 10 && value<=19){
        _display->setCursor(x+2,y+31);
        _display->write(charArray[0]);
        _display->setCursor(x+18,y+31);
        _display->write(charArray[1]);
      }else{
        _display->setCursor(x+2,y+31);
        _display->write(charArray[0]);
        _display->setCursor(x+19,y+31);
        _display->write(charArray[1]);
      }
      
      break;

    case X_LARGE_DICE:
      _display->setFont(&FreeSansBold18pt7b);
      if (value < 10){
        _display->setCursor(x+12,y+33);
        _display->write(charArray);
      }else if (value>= 10 && value<=19){
        _display->setCursor(x+3,y+33);
        _display->write(charArray[0]);
        _display->setCursor(x+18,y+33);
        _display->write(charArray[1]);
      }else{
        _display->setCursor(x+3,y+33);
        _display->write(charArray[0]);
        _display->setCursor(x+21,y+33);
        _display->write(charArray[1]);
      }
      break;
  }
}
#endif

void Dice::drawDiceShape (int x, int y, int size, int radius, bool solid){
  if (solid == false){
    _display->drawRoundRect(x, y, size, size, radius, SSD1306_WHITE);
  }else {
    _display->fillRoundRect(x, y, size, size, radius, SSD1306_WHITE);
  }
}

void Dice::drawDiceDots (int x, int y, int radius, bool solid ){
  if (solid == false){
    _display->drawCircle(x,y,radius,SSD1306_WHITE);
  } else {
    _display->fillCircle(x,y,radius,SSD1306_BLACK);
  }
}
